/**********************************************************************

Filename    :   Direct3DXbox360App.h
Content     :   Simple Xbox 360 Application class
Created     :   
Authors     :   Michael Antonov
Copyright   :   (c) 2005-2006 Scaleform Corp. All Rights Reserved.

This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING 
THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR ANY PURPOSE.

**********************************************************************/

#ifndef INC_D3DXBOX360APP_H
#define INC_D3DXBOX360APP_H

// GFx Includes
#include "GTypes.h"

// ***** XBOX Includes

// XBox title libraries - for input.
#include <xtl.h>


// ***** Application class

class Direct3DXboxApp
{
public:

    // **** Public Interface

    Direct3DXboxApp();
    ~Direct3DXboxApp();
    
    // Creates a window & initializes the application class.
    // Returns 0 if window creation/initialization failed (the app should quit).
    bool            SetupWindow(const char *pname, SInt width, SInt height);

    // Message processing function to be called in the 
    // application loops until it returns 0, to indicate application being closed.
    bool            ProcessMessages();

    // This call cleans up the application and kills the window.
    // If not called explicitly, it will be called from the destructor.
    void            KillWindow();

    // Changes/sets window title
    void            SetWindowTitle(const char *ptitle);

    // Presents the data (swaps the back and front buffers)
    void            PresentFrame();

    // Resets the direct3D, return 1 if successful.
    // On successful reset, this function will call InitRenderer again.
    bool            RecreateRenderer();


    // *** Overrides

    // This override is called from SetupWindow to initialize OpenGL/Direct3D view
    virtual bool    InitRenderer();
    // Should/can be called every frame to prepare the render, user function
    virtual void    PrepareRendererForFrame();

    // Input overrides, invoked during the ProcessMessages call.
    virtual void    OnKey(UInt key, bool downFlag) {}
    virtual void    OnMouseButton(UInt button, bool downFlag, SInt x, SInt y) {}
    virtual void    OnMouseMove(SInt x, SInt y) {}

    // Sizing; by default, re-initalizes the renderer
    virtual void    OnSize(SInt w, SInt h) {}
    // Called when sizing begins and ends.
    virtual void    OnSizeEnter(bool enterSize) {}
    // Handle dropped file
    virtual void    OnDropFiles(char *path) {}

    // *** Simple 2D painting and output, for displaying HUD/Stat overlays

    // Clears the entire back buffer.
    void            Clear(UInt32 color);
    // Initialize and restore 2D rendering view.
    // Push2DRenderView must be called before using 2D functions below,
    // while Pop2DRenderView MUST be called after it is finished.
    void            Push2DRenderView();
    void            Pop2DRenderView();
    // Draw a filled + blended rectangle.
    void            FillRect(SInt l, SInt t, SInt r, SInt b, UInt32 color);
    // Draw a text string (specify top-left corner of characters, NOT baseline)
    void            DrawText(SInt x, SInt y, const char *ptext, UInt32 color);
    // Compute text size that will be generated by DrawText
    void            CalcDrawTextSize(SInt *pw, SInt *ph, const char *ptext);

    // API-independednt toggle for wireframe rendering.
    void            SetWireframe(bool wireframeEnable);


    // *** Implementation


    // Boolean states for the window and app
    bool            Created;
    bool            Active; 
    bool            HideCursor;
    bool            QuitFlag;
    SInt            ExitCode;
    // Set this if the window is to accept drag and drop
    bool            SupportDropFiles;
    bool            SizableWindow;

    // Requested 3D state
    bool            FullScreen;
    bool            FSAntialias;
    UInt            BitDepth;
    // This flag will be set after SetupWindow to indicate that FSAA is available.
    bool            FSAASupported;

    LPDIRECT3D9             pD3D;
    IDirect3DDevice9*       pDevice;
    D3DPRESENT_PARAMETERS   PresentParams;
    // Client size
    int                     Width;
    int                     Height;

    HWND                    hWND;

    // Input state and pad, if any.
    XINPUT_STATE            InputState; 

    // Text/ 2D view rendering support functionality
    bool                    In2DView;
    bool                    WasInScene; 
    D3DVIEWPORT9            ViewportSave;

    // Raw shaders
    IDirect3DVertexShader9* pVShaderCoordCopy;
    IDirect3DPixelShader9*  pPShaderConst;          

    // Viewport un-scale values: first two rows in matrix
    float                   ViewportFactorX[4];
    float                   ViewportFactorY[4];

    // mouse position
    Float                   MouseX, MouseY;
    Float                   MouseXadj, MouseYadj;
};


#endif // INC_D3DXBOX360APP_H
